#!/usr/bin/python3
# -*- coding: utf-8 -*-
########################################################## {COPYRIGHT-TOP} ###
# Licensed Materials - Property of IBM
# PRD0012491
#
# (C) Copyright IBM Corp. 2023
#
# US Government Users Restricted Rights - Use, duplication, or
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
########################################################## {COPYRIGHT-END} ###
import subprocess
import json
import time

class ITDT:
    ''' IBM Tape Dignostic Tool (ITDT) scripting interface class.
    For the detailed list of supported commands and their syntax,
    please refer to the documentation:
    https://www.ibm.com/docs/en/ts4300-tape-library?topic=edition-standard-tapeutil-scripting-commands
    
    Attributes:
    ----------
    itdt_executable : The ITDT program to run.
    '''
    
    version = "0.3.1.20230828"

    def __init__(self, itdt_executable):
        ''' ITDT constructor.
        :param itdt_executable: The path to the ITDT executable program.
        '''
        self.itdt_executable = itdt_executable
        self._device_list = []
    

    def scan_devices(self, verbose=True):
        '''Issue the ITDT scan function
        :param verbose: If True, prints out the list of devices found.
        :return: Return the tape and changer devices.
        '''
        cmd = [self.itdt_executable, 'scanj']
        proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        data, error = proc.communicate()
        #print('Data: '  + data.decode('ascii'))
        #print('Error: '  + error.decode('ascii'))
        #print('code: ' + str(proc.returncode))
        if proc.returncode == 0:
            self._device_list = json.loads(data)

        if verbose and proc.returncode == 0:
            for device in self._device_list:
                #print(device['modelName'] + " --> " + device['serialNumber'])
                print(device)
        
        return self._device_list        

    def run_commands(self, device, commands, return_json=False):
        '''Run a list of ITDT scripting commands.
        
        device : The special device or the generic H-B-T-L string
        commands : List of ITDT scripting commands.
        return_json : If set to True, return the result in JSON.
        return : The command return code and data.
        '''
        cmds = []
        cmds.append(self.itdt_executable)
        cmds.append('-f')
        cmds.append(device)
        cmds += commands.split()
        #cmds = [self.itdt_executable, '-f', device, 'inqj', '0x83']
        proc = None
        data = None
        try:
            proc = subprocess.Popen(cmds, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            data, error = proc.communicate()
            if proc.returncode == 0:
                if data and return_json:
                    data = json.loads(data)
            if error:
                data = error.decode('ascii')      
            
        except (RuntimeError, TypeError, NameError):
            return -1, None

        return proc.returncode, data
    
    
    def move_cartridge(self, device, hdc, source, destination, verbose=False):
        '''ITDT's new move medium command supporting hdc parameter.
        
        device : The special device or the generic H-B-T-L string
        hdc : hdc value.
        source : The source element address to be moved.
        destination : The target address.
        verbose : If set to True, return the result in JSON.
        return : The command result code and diff time.
        '''
        command=f'move -hdc {hdc} {source} {destination}'
        if verbose:
            print("Move command: ", command)

        start_time = time.time()        
        result, data = self.run_commands(device, command)
        diff_time = time.time() - start_time
        if result == 0:
            if verbose:
                print(data)
        else:   
            print(f"Error while move cartridge {source}")

        return result, diff_time



class ITDTROS(ITDT):
    ''' IBM Tape Dignostic Tool (ITDT) scripting interface class for REST Over SCSI.
    RoS (REST over SCSI) is used with the TS4500 and Diamondback libraries.
    For the detailed list of supported commands and their syntax,
    please refer to the documentation:
    https://www.ibm.com/docs/en/ts4300-tape-library?topic=commands-ts4500-rest-over-scsi
    
    Attributes:
    ----------
    itdt_executable : The ITDT program to run.
    device_file_changer : The special device or the generic H-B-T-L string of the TS4500/diamonback library.
    device_file_drive : The special device or the generic H-B-T-L string of the tape device.
    '''

    def __init__(self, itdt_executable, device_file_changer, device_file_drive=None):
        ITDT.__init__(self, itdt_executable)
        self.cartridges = []
        self.cartridge_count=0
        self.slot = {}
        self.device_changer = device_file_changer
        self.device_drive   = device_file_drive
        self.tier=9
    
    def load_slot_info(self, slot, verbose=False):
        ''' Get the slot information for a dedicated slot.
        slot : The slot id.
        verbose : Print out the information for a dedicated slot.
        return : The result code and data.
        '''
        #print('Load slot info ' + slot)
        result, data = self.run_commands(self.device_changer, 'ros GET /v1/slots/'+ slot, True)
        if result == 0:
            self.cartridges = data[0]['contents']
            self.slot = data[0] 
            self.cartridge_count=len(self.cartridges)-self.cartridges.count(None)           
            self.tier = len(self.cartridges)
            if verbose:
                print(f"Cartridge count for slot: {slot} is {self.cartridge_count}")
        else:   
            print(f"Error while reading slot info {slot}")            

        return result, data


    def load_data_cartridge_info(self, volser, verbose=False):
        ''' Get the data cartridge for a dedicated volser id.
        volser : The volser id.
        verbose : Print out the information for a dedicated slot.
        return : The result code and data.
        '''
        #print('Load volser info ' + volser)
        result, data = self.run_commands(self.device_changer, 'ros GET /v1/dataCartridges/'+ volser, True)
        if verbose:
            print(data)            
        
        return result, data

    
    def get_drive_info(self, serial, verbose=False):
        ''' Get the drive info.
        volser : The volser id.
        verbose : Print out the information for a dedicated slot.
        return : The result code and data.
        '''        
        if not isinstance(serial, str):
            return -1, None
        result, data = self.run_commands(self.device_changer, 'ros GET /v1/drives/'+ serial, True)
        if verbose:
            print(data)            
        
        return result, data

    
    def check_slot(self, logical_library, slot):
        ''' Check a slot for cartridges might no associated with the logical_library.
        logical_library : The logical tape library name to be compared.
        slot : The library slot to be verified.
        return : The number of cartridges in this slot not associated with the logical_library.
        ''' 
        result, data = self.load_slot_info(slot)
        if result != 0:
            #print ("Load slot info failed")
            return result, None
        
        if self.cartridge_count >= 0:
            count = 0            
            for volser in self.cartridges:
                if volser is None:
                    continue
                result, data = self.load_data_cartridge_info(volser)
                if result == 0 and len(data) == 1:
                    #print(f"Volser: {volser}, Elementaddress: {data[0]['elementAddress']}, internalAddress: {data[0]['internalAddress']}")
                    ll = data[0]['logicalLibrary']
                    if ll != logical_library:
                        print (f"LogicalLibray {logical_library} does not match for {data[0]}")
                        count+=1
            if count > 0:
                print ("!!! Either select an other slot or move the cartridges.")                        
            return count


    def find_data_cartridges(self, logical_library):
        ''' Get the cartridges within one logical library.
        logical_library : The logical tape library name to be compared.
        return : None | The list of cartridges.
        '''        
        result, data = self.run_commands(self.device_changer, 'ros GET /v1/dataCartridges', True)
        ret = []
        if result == 0:            
            #print (data)            
            for cartridge in data:
                #print (cartridge)
                if cartridge['logicalLibrary'] == logical_library and cartridge['state'] == 'normal' \
                and (cartridge['location'].endswith('T1') or cartridge['location'].endswith('T2')):
                    if cartridge['volser'] not in self.cartridges:
                        ret.append(cartridge['volser'])
            return ret
        else:
            return None        

    
    def move_to_slot(self, slot, volume, verbose=False):
        ''' Move a dedicated volume to a given slot
        volser : The volser id.
        verbose : Print out the information for a dedicated slot.        
        '''        
        # Check first, if the tier is 1 or 2, otherwise the moveToSlot will fail
        result, data = self.load_data_cartridge_info(volume, False)
        if result == 0:
            location = data[0]['location']
            if location.endswith('T1') or  location.endswith('T2'):
                parameter = '{"type":"moveToSlot", "cartridge":"' + volume + '", "destinationLocation":"' + slot + '"}'
                result, data = self.run_commands(self.device_changer, 'ros POST /v1/workItems ' + parameter, True)
                ret = []
                if result == 0 and verbose:
                    print(f"Cartridge: {volume} moved to slot: {slot}")
                else:
                    print(f"Move failed: {data}")

    def prepare_slot_for_test(self, logical_library, slot, verbose=False, headless=False):
        ''' Prepare a slot with cartridges. I.e.fill up all tiers with cartridges belonging to the same logical library.
        logical_library : The logical tape library name to be compared.
        slot : The library slot to be filled up.
        verbose : Print out the information.
        headless : Use cartridges found in inventory to be moved into it w/o users dialog.        
        '''
        if self.check_slot(logical_library, slot) == 0:     # Check for cartridges from other LL's 
            self.load_slot_info(slot)                       # Get the numbers for a dedicated slot
            print(f"Slot: {slot} has {self.cartridge_count} cartridge(s) loaded")
            current_cartridges = self.cartridge_count
            if self.cartridge_count < self.tier:
                answer='y'
                #print(f'Slot: {slot}, cartridges: {self.cartridges}')
                if not headless:            
                    answer = input(f'Should be other cartridges loaded to slot {slot} to fill up(y/n)? ')
                if answer.casefold()=="y":
                    print ('Loading cartridges to slot: ', slot)
                    carts = self.find_data_cartridges(logical_library)
                    missing = self.tier - self.cartridge_count
                    if len(carts) > missing:
                        print (carts)
                        #Moving cartridges into slot - fill up all tiers
                        for x in range(0, missing):
                            self.move_to_slot(slot+'T1', carts[x], verbose)
                            current_cartridges += 1
                            time.sleep(1)
                    else:
                        print("No cartridge available (in TIER 0/1) for logical library {logical_library} to get loaded into {slot}")
                                
                # Synchronize the move_to_slot calls - wait till move is completed
                loop_counter = 10
                while (current_cartridges > self.cartridge_count) and (loop_counter > 0):
                    time.sleep(5)
                    self.load_slot_info(slot)   #update numbers in class members    
                    loop_counter -= 1
        else:
            print(f"Slot {slot} is fully loaded.")

    def mount_and_unmount_cartridge(self, drive, cart_number=None, hdc=None):
        ''' Mount and un-mount a cartridge and measure the 2 times.
        drive : The tape drive used for mount/unmount.
        cart_number : The number of a cartridge (tier level) 1-9.
        hdc : HD control value either 0|1|2|3.
        return : Return code (0|-1|-2) and if return code is 0, the mount and unmount time.
        '''
        while cart_number is None:
            cart_number = int(input(f'Select cartridge number to load 1 to {self.cartridge_count}: '))
            if (cart_number < 1 and cart_number > self.cartridge_count):
                cart_number=None
        
        while hdc not in range(3):
            hdc = int(input('Select hdc value for move medium (0|1|2|3): '))        
        if hdc < 0:
            hdc=0

        volume = self.cartridges[cart_number-1]
        
        result, cartridge = self.load_data_cartridge_info(volume)
        if result == 0:            
            source_address = cartridge[0]['elementAddress']
            drive_element_address = drive['elementAddress']
            drive_sn = drive['sn']
            print (f'  Moving cartridge {volume}[{source_address}] to drive {drive_sn}[{drive_element_address}]...', end='',  flush=True)
            result,mount = self.move_cartridge(self.device_changer, hdc, source_address, drive_element_address, False)
            print(f'{mount:.2f}')
            if result == 0:
                if self.device_drive:
                    print(f'  Unloading cartridge from drive {drive_sn}...', end='', flush=True)
                    self.run_commands(self.device_drive, 'load tur tur') #Catching sk/asc/ascq 06/28/00 ...MEDIUM MAY HAVE CHANGED 
                    result, data = self.run_commands(self.device_drive, 'unload')
                    if result == 0:
                        print('Done')
                    else:
                        print(f'Failed with {result}--> {data}')
                
                print (f'  Moving cartridge {volume}[{source_address}] from drive {drive_sn}[{drive_element_address}] back...', end='', flush=True)
                result,umount = self.move_cartridge(self.device_changer, 0, drive_element_address, source_address, False)
                print(f'{umount:.2f}')
                if result != 0:
                    print(f'Move back failed - {drive_element_address} to {source_address}')    
                    return -2, None, None
            else:
                print(f'Move failed - {source_address} to {drive_element_address}')
                return -1, None, None
            return 0, mount, umount

