#!/usr/bin/python3
# -*- coding: utf-8 -*-
########################################################## {COPYRIGHT-TOP} ###
# Licensed Materials - Property of IBM
# PRD0012491
#
# (C) Copyright IBM Corp. 2023
#
# US Government Users Restricted Rights - Use, duplication, or
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
########################################################## {COPYRIGHT-END} ###
import argparse
import platform
import os
from pathlib import Path
import itdt_script as itdt
INDENT =25
#Arg parser
def parseArguments():
    parser = argparse.ArgumentParser(
        prog='DiamonBack mount time check',
        description='IBM diamondBack test script to evaluate mount/un-mount performance.',)
    parser.add_argument('-c','--changer', type=str, required=True,
                        help='The special device file of the TS4500/DiamonBack used.')
    parser.add_argument('-d',"--drive", type=str, required=True,
                        help='The drive special file assigned to the library used to mount cartridges.')
    parser.add_argument('-s','--slot', type=str, required=True,
                        help='The slot to be used for cartridges.')
    parser.add_argument('-n', '--number', type=int, required=False, help='The TIER(cartridge) number to get loaded.')
    parser.add_argument('-hdc', type=int, required=False, help='The HD Control value for the ITDT move medium command.')
    parser.add_argument('-v', '--verbose', action='count', dest='verbose', default=0,
                    help="verbose output (repeat for increased verbosity)")
    return parser.parse_args()

# print current settings to console
def printSettings():
    print('{:.<25} {}'.format('Version',itdt.ITDT.version))
    print('{:.<25} {}'.format('OS', platform.system())) 
    print('{:.<25} {}'.format('OS Version', platform.version()))
    print('{:.<25} {}'.format('OS Release', platform.release()))
    print('{:.<25} {}'.format('Machine Type', platform.machine()))
    print('{:.<25} {}'.format('Script Path ', script_path))
    print('{:.<25} {}'.format('ITDT Path ', itdt_path))
    print('{:.<25} {}'.format('Verbose Level',args.verbose))
    print('{:.<25} {}'.format('TS4500/DiamondBack device', args.changer))
    print('{:.<25} {}'.format('Tape Drive device', args.drive))
    print('{:.<25} {}'.format('Slot', args.slot))

# initial globals
def initialize():
    global script_path
    global itdt_path
    script_path=os.path.realpath(os.path.dirname(__file__))    
    itdt_path =Path(script_path+ '/../itdt').resolve()

# here starts Main()
args=parseArguments()
initialize()
printSettings()

changer_device = args.changer
drive_device = args.drive
slot = args.slot
drive_serial=''
cart_number = args.number
hdc = args.hdc
headless=False
if cart_number:
    headless=True

diamondBack = itdt.ITDTROS(itdt_path, changer_device, drive_device)

print(f'*********** Start of script *************  ')
res=None 
dat=None

# Issue a inqj (inquiry in JSON format) to get the serial number
# needed for the REST over SCSI (ros) functions below
res, dat = diamondBack.run_commands(drive_device, 'inqj 0x83', True)
if res != 0:
    print (f'Drive {drive_device} could not be opened: {dat}')   
    quit(-1)
else:
    drive_serial = dat['data']['serialNumberDrive']
    print(f'Drive serial numer is: {drive_serial}')

# Get the information of the drive and the name of the logical library
res,dat = diamondBack.get_drive_info(drive_serial)
if res != 0:
    print (f'Drive {drive_serial} info failed: {dat}')    
else:    
    drive_info = dat[0]
    ll = drive_info['logicalLibrary']
    diamondBack.load_slot_info(slot)
    diamondBack.prepare_slot_for_test(ll, slot, True, headless) # fill up cartridges into slot
    print(f'Slot: {slot}, cartridges before test: {diamondBack.cartridges}')

    res, mount_time, unmount_time = diamondBack.mount_and_unmount_cartridge(drive_info, cart_number, hdc)
    if(mount_time):
        print(f'Mount-time: {mount_time:.2f}')
    if(unmount_time):
        print(f'Unmount-time:{unmount_time:.2f}')

    #cleanup, collect the cartridges
    #print('Restore cartridges....')
    for volser in reversed(diamondBack.cartridges):
        if volser is None:
            continue
        #print (f'  Restore cartridge {volser}')
        res, dat = diamondBack.load_data_cartridge_info(volser)
        if res==0 and len(dat)==1:
            if slot and slot not in dat[0]['location']:
                diamondBack.move_to_slot(slot+'T1', volser, True)
                
    
    diamondBack.load_slot_info(slot)
    #print(f'Slot: {slot_name}, cartridges after test: {itdt.cartridges}')



